param([string]$InstanceName, [int]$TimePeriod, [int]$FailuresCountThreshold, [string]$CustomSQLName, [string]$Exclude)

# Escape single quote (just in case)
if ($Exclude -match "'") {
    $Exclude = $Exclude -replace "'", "''"
}

if ($Exclude -match "#") {
    
    [array]$ExcludeArray = $Exclude -split "#"
    [array]$Exclude = @()

    # Transform exclude list of jobs with delimeter into an array with elements each enclosed in single quotes
    Try {
        foreach ($Element in $ExcludeArray) {
            # Enclose each job name in single quotes
            $Exclude += "'$Element'"
        }

        # Create a string that will be inserter in query
        [string]$ExcludeString = $Exclude -join ","
    }
    Catch {
        $Exception = $_.Exception
        Throw "Failed to transform exclude list:`n$($Exception.Message)"
    }
}
else {
    $ExcludeString = "'$Exclude'"
}

if ($TimePeriod -gt 0) {
    $StrLength = 4096

    if ($CustomSQLName) { $ComputerName = $CustomSQLName }
    else { $ComputerName = (Get-WmiObject Win32_Computersystem).name }

    if ($InstanceName -like "SQLServer") { $InstanceName = "" }
    else {
        $InstanceName = $InstanceName.Replace('MSSQL$', "")
        $InstanceName = $InstanceName.Replace("'", "")
    }

    $ConnectionString = $ComputerName + '\' + $InstanceName

    $Query = @"
DECLARE @TimeIntervalMinutes int = $TimePeriod
DECLARE @FailuresCountThreshold int = $FailuresCountThreshold

-- Light
;with h as
(
select
job_id,
step_id,
step_name,
DATEADD(hh, run_time/10000, DATEADD(mi, run_time%10000/100, DATEADD(ss, run_time%100, CONVERT(datetime, CAST(run_date as varchar(8)))))) as run_date,
run_duration/10000 as run_duration_hh,
run_duration%10000/100 as run_duration_mi,
run_duration%100 as run_duration_ss
from
msdb.dbo.sysjobhistory
where run_status = 0
)
, b as
(
select
j.name as job_name,
s.step_id,
h.step_name,
h.run_date,
DATEADD(hh, h.run_duration_hh, DATEADD(mi, h.run_duration_mi, DATEADD(ss, h.run_duration_ss, h.run_date))) as end_date
from
h
JOIN msdb.dbo.sysjobs j ON h.job_id = j.job_id
JOIN msdb.dbo.sysjobsteps s ON j.job_id = s.job_id AND h.step_id = s.step_id
join msdb.dbo.sysjobschedules js on j.job_id = js.job_id
join msdb.dbo.sysschedules sc on js.schedule_id = sc.schedule_id
where j.enabled = 1 and sc.enabled = 1
  and not (sc.freq_type > 4 or (sc.freq_type = 4 and sc.freq_interval = 1 and sc.freq_subday_type = 1))
)
select
job_name,
step_name,
CONVERT(varchar(20), min(run_date), 120) as first_failure_date,
CONVERT(varchar(20), max(run_date), 120) as last_failure_date,
--DATEDIFF(hh, min(run_date), max(run_date)) as failures_interval_hours,
--DATEDIFF(mi, min(run_date), max(run_date)) as failures_interval_minutes,
count(1) as failure_num
from
b
where DATEDIFF(mi, end_date, GETDATE()) <= @TimeIntervalMinutes
and job_name not in ($ExcludeString)
group by job_name, step_name
having count(1) >= @FailuresCountThreshold
order by job_name, step_name
"@
    try {
        $Datatable = New-Object System.Data.DataTable
    
        $Connection = New-Object System.Data.SQLClient.SQLConnection
        $Connection.ConnectionString = "server=$ConnectionString;database='master';trusted_connection=true;"
        $Connection.Open()
        $Command = New-Object System.Data.SQLClient.SQLCommand
        $Command.Connection = $Connection
        $Command.CommandText = $Query
        $Reader = $Command.ExecuteReader()
        $Datatable.Load($Reader)
        $Connection.Close()
 

        if (($Datatable.rows).count -ge 1) {
            $FormatedData = $Datatable.Rows | select-object job_name, step_name, first_failure_date, last_failure_date, failure_num | FL | out-string
            if ($FormatedData.Length -gt $StrLength) { $FormatedData = $FormatedData.substring(0, $StrLength) }
            write-output $FormatedData
        }

        else { write-output "Result: none" }
    }
    Catch {
        write-output "Result: none"
        $ConnectionString
        $Exception = $_.Exception.Message
        if ($Exception.Length -gt $StrLength) { $Exception = $Exception.substring(0, $StrLength) }
        write-output $Exception
    }
}
else { write-output "Result: none" }
