param([string]$InstanceNames, [int]$TimePeriod, [string]$Database)

# Escape single quote (just in case)
if ($Exclude -match "'") {
    $Exclude = $Exclude -replace "'", "''"
}

if ($TimePeriod -gt 0) {
    $StrLength = 4096

    if ($CustomSQLName) { $ComputerName = $CustomSQLName }
    else { $ComputerName = (Get-WmiObject Win32_Computersystem).name }

    if ($InstanceName -like "SQLServer") { $InstanceName = "" }
    else {
        $InstanceName = $InstanceName.Replace('MSSQL$', "")
        $InstanceName = $InstanceName.Replace("'", "")
    }

    $ConnectionString = $ComputerName + '\' + $InstanceName

    $Query = @"
CREATE TABLE #DBFileSize (
  database_id int, 
  file_id int, 
  InternalDBSpaceTotal_MB decimal(18,2),
  InternalDBSpaceUsed_MB  decimal(18,2),
  AutoGrowth tinyint
);

insert into #DBFileSize(database_id, file_id, InternalDBSpaceTotal_MB, InternalDBSpaceUsed_MB, AutoGrowth)
exec sp_msforeachdb 
'
use [?]; 
select
  fs.database_id,
  df.file_id,
  cast(df.size / 128. as decimal(18,2)) InternalDBSpaceTotal_MB,
  cast(fs.allocated_extent_page_count * 8.0 / 1024. as decimal(18,2)) InternalDBSpaceUsed_MB,
  cast(case when df.growth = 0 then 0 else 1 end as tinyint) AutoGrowth
from sys.dm_db_file_space_usage fs
join sys.database_files df on fs.file_id = df.file_id
where df.type = 0
';

;with a as(
    select t.database_id, t.file_id, t.InternalDBSpaceTotal_MB, t.InternalDBSpaceUsed_MB, t.AutoGrowth,
    vs.volume_mount_point Drive,
    vs.logical_volume_name,
    cast(vs.total_bytes / 1024. / 1024. as decimal(18,2)) DriveSpaceTotal_MB,
    cast((vs.total_bytes - vs.available_bytes) / 1024. / 1024. as decimal(18,2)) DriveSpaceUsed_MB
  from #DBFileSize t
  cross apply sys.dm_os_volume_stats(t.database_id, t.file_id) vs
  where t.database_id > 4
  ), b as (
  select
    database_id, Drive, logical_volume_name, DriveSpaceTotal_MB, DriveSpaceUsed_MB,
    sum(InternalDBSpaceTotal_MB) InternalDBSpaceTotal_MB,
    sum(InternalDBSpaceUsed_MB)  InternalDBSpaceUsed_MB,
    max(AutoGrowth) AutoGrowth
  from a
  group by database_id, Drive, logical_volume_name, DriveSpaceTotal_MB, DriveSpaceUsed_MB
  )

  select
  db_name(database_id) AS DbName, Drive, logical_volume_name, DriveSpaceTotal_MB, DriveSpaceUsed_MB, DriveSpaceTotal_MB - DriveSpaceUsed_MB DriveSpaceFree_MB,
  sum(InternalDBSpaceTotal_MB) InternalDBSpaceTotal_MB,
  sum(InternalDBSpaceUsed_MB)  InternalDBSpaceUsed_MB,
  sum(InternalDBSpaceTotal_MB - InternalDBSpaceUsed_MB) InternalDBSpaceFree_MB,
  sum(InternalDBSpaceTotal_MB - InternalDBSpaceUsed_MB) + case when AutoGrowth = 0 then 0 else DriveSpaceTotal_MB - DriveSpaceUsed_MB end TotalSpaceFree_MB
from b
group by db_name(database_id), Drive, logical_volume_name, DriveSpaceTotal_MB, DriveSpaceUsed_MB, AutoGrowth
order by Drive, DbName;

DROP TABLE #DBFileSize;
"@
try {
    $Datatable = New-Object System.Data.DataTable

    $Connection = New-Object System.Data.SQLClient.SQLConnection
    $Connection.ConnectionString = "server=$ConnectionString;database='master';trusted_connection=true;"
    $Connection.Open()
    $Command = New-Object System.Data.SQLClient.SQLCommand
    $Command.Connection = $Connection
    $Command.CommandText = $Query
    $Reader = $Command.ExecuteReader()
    $Datatable.Load($Reader)
    $Connection.Close()


    if (($Datatable.rows).count -ge 1) 
    {
        $FormatedData = $Datatable.Rows | FL | out-string
        if($FormatedData.Length -gt $StrLength) {$FormatedData = $FormatedData.substring(0, $StrLength)}
        write-output $FormatedData
    }
    else { write-output "Result: none" }
}
Catch {
    write-output "Result: none"
    $ConnectionString
    $Exception = $_.Exception.Message
    if ($Exception.Length -gt $StrLength) { $Exception = $Exception.substring(0, $StrLength) }
    write-output $Exception
}
}
else { write-output "Result: none" }